import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Globe, Languages, BookOpen, Award, Sparkles, ArrowRight } from 'lucide-react'
import { motion } from 'framer-motion'
import './EnrollCourse.css'

const EnrollCourse = () => {
  const navigate = useNavigate()
  const [selectedLanguage, setSelectedLanguage] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const languages = [
    { 
      id: 'english', 
      name: 'Английский', 
      iconType: 'globe',
      iconComponent: Globe,
      color: '#3B82F6',
      description: 'Самый популярный язык в мире',
      features: ['Грамматика', 'Разговорная речь', 'Бизнес-английский']
    },
    { 
      id: 'spanish', 
      name: 'Испанский', 
      iconType: 'languages',
      iconComponent: Languages,
      color: '#EF4444',
      description: 'Язык страсти и культуры',
      features: ['Испанская культура', 'Латинская Америка', 'Литература']
    },
    { 
      id: 'chinese', 
      name: 'Китайский', 
      iconType: 'book',
      iconComponent: BookOpen,
      color: '#F59E0B',
      description: 'Откройте двери в Азию',
      features: ['Иероглифы', 'Тоновая система', 'Культура Китая']
    },
    { 
      id: 'german', 
      name: 'Немецкий', 
      iconType: 'award',
      iconComponent: Award,
      color: '#10B981',
      description: 'Язык бизнеса и науки',
      features: ['Деловой немецкий', 'Научная лексика', 'Европейская культура']
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedLanguage && selectedDate) {
      const savedCourses = localStorage.getItem('languageCourses')
      const courses = savedCourses ? JSON.parse(savedCourses) : []
      
      const languageData = languages.find(l => l.id === selectedLanguage)
      const newCourse = {
        id: Date.now(),
        language: languageData.name,
        level: 'Beginner',
        progress: 0,
        rating: 0,
        iconType: languageData.iconType
      }

      courses.push(newCourse)
      localStorage.setItem('languageCourses', JSON.stringify(courses))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedLanguage('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll-page">
      <div className="enroll-split">
        <motion.div 
          className="enroll-visual"
          initial={{ opacity: 0, x: -50 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="visual-content">
            <motion.div 
              className="visual-icon-wrapper"
              initial={{ scale: 0, rotate: -180 }}
              animate={{ scale: 1, rotate: 0 }}
              transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
            >
              <Sparkles size={80} strokeWidth={2} className="visual-icon" />
            </motion.div>
            <h1 className="visual-title">Начните изучение языка</h1>
            <p className="visual-description">
              Выберите язык и дату старта. Наши преподаватели помогут вам достичь ваших целей
            </p>
            <div className="visual-features">
              <div className="feature-item">
                <CheckCircle size={20} strokeWidth={2} />
                <span>Индивидуальный подход</span>
              </div>
              <div className="feature-item">
                <CheckCircle size={20} strokeWidth={2} />
                <span>Гибкий график</span>
              </div>
              <div className="feature-item">
                <CheckCircle size={20} strokeWidth={2} />
                <span>Опытные преподаватели</span>
              </div>
            </div>
          </div>
        </motion.div>

        <motion.div 
          className="enroll-form-section"
          initial={{ opacity: 0, x: 50 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <form className="enroll-form-card" onSubmit={handleSubmit}>
            <div className="form-header">
              <h2 className="form-title">Записаться на курс</h2>
              <p className="form-subtitle">Заполните форму для начала обучения</p>
            </div>

            <div className="form-content">
              <div className="form-section">
                <label className="section-label">
                  <Globe size={18} strokeWidth={2} />
                  Выберите язык
                </label>
                <div className="languages-compact">
                  {languages.map(language => (
                    <motion.button
                      key={language.id}
                      type="button"
                      className={`language-compact ${selectedLanguage === language.id ? 'selected' : ''}`}
                      onClick={() => setSelectedLanguage(language.id)}
                      whileHover={{ scale: 1.05 }}
                      whileTap={{ scale: 0.95 }}
                      style={{ 
                        borderColor: selectedLanguage === language.id ? language.color : 'rgba(99, 102, 241, 0.2)',
                        background: selectedLanguage === language.id ? `${language.color}10` : 'rgba(255, 255, 255, 0.9)'
                      }}
                    >
                      <div className="compact-icon" style={{ background: `${language.color}15` }}>
                        <language.iconComponent size={24} strokeWidth={2} color={language.color} />
                      </div>
                      <div className="compact-info">
                        <span className="compact-name">{language.name}</span>
                        <span className="compact-desc">{language.description}</span>
                      </div>
                      {selectedLanguage === language.id && (
                        <motion.div 
                          className="compact-check"
                          initial={{ scale: 0 }}
                          animate={{ scale: 1 }}
                          style={{ background: language.color }}
                        >
                          <CheckCircle size={16} strokeWidth={2} color="white" />
                        </motion.div>
                      )}
                    </motion.button>
                  ))}
                </div>
              </div>

              <div className="form-section">
                <label htmlFor="date" className="section-label">
                  <Calendar size={18} strokeWidth={2} />
                  Дата старта курса
                </label>
                <input
                  type="date"
                  id="date"
                  className="date-input"
                  value={selectedDate}
                  onChange={(e) => setSelectedDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="enroll-submit" 
                disabled={!selectedLanguage || !selectedDate}
                whileHover={{ scale: 1.02 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на курс</span>
                <ArrowRight size={20} strokeWidth={2} />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="submit-success"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={24} strokeWidth={2} />
                  <span>Вы успешно записались на курс!</span>
                </motion.div>
              )}
            </div>
          </form>
        </motion.div>
      </div>
    </div>
  )
}

export default EnrollCourse
